
/*
 * sysinfo.c            - get kernel info
 *
 */

/* 
 * Written by Gabor Herr <herr@iti.informatik.th-darmstadt.de>.
 *
 * Copyright (c) 1992, 1993 by Gabor Herr, all rights reserved.
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that may name is not used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission. I make no representations
 * about the suitability of this software for any purpose. It is
 * provided "as is" without express or implied warranty.
 */

//
// modified in 1999, 2000, 2001 by Jason Hildebrand for use with gsysinfo 
//

/* $Id: sysinfo.c,v 1.3 2002/01/30 17:23:09 jdhildeb Exp $ */
 
#include "sysinfo.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>

static int meminfo_fd;
static int load_fd;
static int loadavg_fd;
static int swaps_fd;
static int apm_fd;
static int net_fd;
#if defined( DEBUG )
extern FILE * debugfile;
#endif

static char buffer[1024];

// read a file from the given descriptor into the buffer
static void reread( int fd )
{
        ssize_t len;
        int    rc;

        rc = lseek( fd, 0L, 0 );
        if( rc != 0 ) {
            return;
        }
        len = read( fd, buffer, sizeof(buffer) - 1 );
        if( len <= 0 ) {
            return;
        };
        buffer[ len ] = '\0';   // stick a null char at the end of the file, so that we can
                                // test for it later
        return;
}

// parse  the meminfo file, and assign the values we need into the 
// meminfo struct
static int parse_meminfo( struct meminfo * result ) {
    char * bufptr = strchr( buffer, '\n' );
    char * endptr;
    int numvalues = 0;
    unsigned long * target;
    unsigned long value;

    bufptr++;  // advance past the newline

    // now try to recognize one of the tags: MemTotal, MemFree, MemShared,
    // Buffers, Cached, SwapTotal, SwapFree very quickly
    while( *bufptr != '\0' && numvalues < 7 ) {
        target = NULL;
        switch( *bufptr ) {
        case 'M':
            switch( bufptr[ 3 ] ) {
            case 'T':
            if( strncmp( bufptr, "MemTotal:", 9 ) == 0 ) {
                bufptr += 9;
                target = &result->total;
            }
            break;
            case 'F':
            if( strncmp( bufptr, "MemFree:", 8 ) == 0 ) {
                bufptr += 8;
                target = &result->free;
            }
            break;
            case 'S':
            if( strncmp( bufptr, "MemShared:", 10 ) == 0 ) {
                bufptr += 10;
                target = &result->shared;
            }
            break;
            default:
            break;
            }
            break;
        case 'B':
            if( strncmp( bufptr, "Buffers:", 8 ) == 0 ) {
            bufptr += 8;
            target = &result->buffers;
            }
            break;
        case 'C':
            if( strncmp( bufptr, "Cached:", 7 ) == 0 ) {
            bufptr += 7;
            target = &result->cache;
            }
            break;
        case 'S':
            if( bufptr[ 4 ] == 'T' ) {
            if( strncmp( bufptr, "SwapTotal:", 10 ) == 0 ) {
                bufptr += 10;
                target = &result[1].total;
            }
            } else if( bufptr[ 4 ] == 'F' ) {
            if( strncmp( bufptr, "SwapFree:", 9 ) == 0 ) {
                bufptr += 9;
                target = &result[1].free;
            }
            }
            break;
        default:
            break;
        }

        if( target != NULL ) {
            // if we found a value that we want, then read it and assign it
            // to the target struct member
            value = strtoul( bufptr, &endptr, 10 );
            if( bufptr != endptr ) {
            *target = value;
            numvalues += 1;
            }
        }
        bufptr = strchr( bufptr, '\n' );
        bufptr++;
    }
    return( numvalues - 7 );
}

void get_meminfo( int *swapdevs, struct meminfo *result )
{
        int i;
        char *bufptr;
        int res;
        int sw_size, sw_used;
        int rc;

        reread( meminfo_fd );

        *swapdevs = 1;
    
        rc = parse_meminfo( result );

        /* if rc is 0, then we found all the fields.  Then this is the
         * new /proc/meminfo format */
        if( rc == 0 ) {
            if (swaps_fd > 0) {

            reread( swaps_fd );
            bufptr = buffer;

            for ( i = 1; i < MAX_SWAPFILES; i++ ) {

                bufptr = strchr( bufptr, '\n' );
                bufptr++;

                if ( *bufptr == '\0' ) {
                    break;
                }
                res = sscanf( bufptr, "%*s %*s %d %d", &sw_size, &sw_used);
                if( res != 2 ) {
                    break;
                }

                result[i].total = sw_size;
                result[i].free = sw_size - sw_used;
            }

            *swapdevs = i - 1;  
            }
        } else {
            /* Fall back to old /proc/meminfo file format */
            bufptr = strchr( buffer, '\n' ) + 1;
            sscanf( bufptr, "Mem: %u %*u %u %u %u",
                &result[0].total,
                &result[0].free,
                &result[0].shared,
                &result[0].cache );
            result[0].buffers = 0;

            for ( i = 1; i < MAX_SWAPFILES; i++ ) {
                bufptr = strchr( bufptr, '\n' ) + 1;
                if ( *bufptr == '\0' ||
                 (res = sscanf( bufptr, "Swap: %u %*u %u",
                        &result[i].total,
                        &result[i].free )) != 2 )
                    break; 
            }
            *swapdevs = i - 1;  
        }
}

double get_load( void )
{
        double load;

        reread( loadavg_fd );
        sscanf( buffer, "%lf", &load );
        return load;
}

void get_cpu(struct load * result)
{
        static struct load last_load = { 0, 0, 0, 0, 0 };
        struct load curr_load;

        reread( load_fd );
        sscanf( buffer, "%*s %lu %lu %lu %lu\n", &curr_load.user,
            &curr_load.nice, &curr_load.system, &curr_load.idle );
        curr_load.total = curr_load.user + curr_load.nice + curr_load.system +
            curr_load.idle;

        result->total  = curr_load.total  - last_load.total + 1;
        result->user   = curr_load.user   - last_load.user;
        result->nice   = curr_load.nice   - last_load.nice;
        result->system = curr_load.system - last_load.system;
        result->idle   = curr_load.idle   - last_load.idle;
        last_load.total  = curr_load.total;
        last_load.user   = curr_load.user;
        last_load.nice   = curr_load.nice;
        last_load.system = curr_load.system;
        last_load.idle   = curr_load.idle;
}

void get_apm_status( struct apminfo * info )
{
    char    timeunits[10];
    int            res;

    if( apm_fd != 0 ) {
        reread( apm_fd );
        info->has_apm = 1;
        res = sscanf( buffer, "%*f %*f %*x %x %*x %*x %d%% %d %s", 
                  &info->acstatus,
                  &info->battery_perc, 
                  &info->battery_time,
                  timeunits );
        //printf( "%x %d %d %s\n", 
                //info->acstatus, 
                //info->battery_perc, 
                //info->battery_time, 
                //timeunits );
        if( strncmp( timeunits, "min", 3 ) == 0 ) {
            info->battery_time *= 60;
        }
    } else {
        info->has_apm = 0;
    }
}

int get_net_status( struct netinfo * info )
{
    char *  bufptr;
    char *  dst;
    int            res;
    int            i;

    reread( net_fd );
    bufptr = strchr( buffer, '\n' ) + 1;    // skip the first line
    for( i = 0; i < MAX_NET_DEVS; i++ ) {
        // goto start of next line
        bufptr = strchr( bufptr, '\n' ) + 1;
        
        if( *bufptr == '\0' ) {
            break;
        }

        // skip whitespace
        while( *bufptr == ' ' ) {
            bufptr++;
        }

        // copy characters until the next colon ':'.
        // this is the interface name
        dst = info[i].iface;
        while( *bufptr != ':' ) {
            *dst = *bufptr;
            dst++;
            bufptr++;
        }
        *dst = '\0';    
        
        bufptr++;
        info[i].old_received_bytes = info[i].received_bytes;
        info[i].old_sent_bytes = info[i].sent_bytes;
        res = sscanf( bufptr, "%lu %*d %*d %*d %*d %*d %*d %*d %lu", 
                  &info[i].received_bytes,
                  &info[i].sent_bytes );
    }
    return( i );
}

int sysinfo_init( void )
{
    int rc;

    if ((meminfo_fd = open("/proc/meminfo",O_RDONLY)) < 0) {
        perror("/proc/meminfo");
        return 1;
    }
    if ((loadavg_fd = open("/proc/loadavg",O_RDONLY)) < 0) {
        perror("/proc/loadavg");
        return 1;
    }
    if ((load_fd = open("/proc/stat",O_RDONLY)) < 0) {
        perror("/proc/stat");
        return 1;
    }

    /* Failing here is not critical. We can get the info from meminfo. */
    swaps_fd = open("/proc/swaps",O_RDONLY);

    // check if apm exists; and if so open apm file
    struct stat buf;
    apm_fd = open("/proc/apm", O_RDONLY );
    rc = fstat( apm_fd, &buf );
    if( rc != 0 ) {
        apm_fd = 0;
    }

    net_fd = open("/proc/net/dev", O_RDONLY );
    return 0;
}


#if 0 

// this main is just used for initial testing of the procedures 
// in this file

int main() 
{
    int rc;
    int i;
    struct netinfo info[10];
    struct apminfo apm;

    rc = sysinfo_init();
    if( rc != 0 ) {
        return 1;
    }
    rc = get_net_status( info );
    for( i = 0; i < rc; i++ ) {
        printf( "Iface: %s, Sent: %d, Recv: %d\n", info[i].iface, info[i].sent_bytes,
                info[i].received_bytes );
    }

    get_apm_status( &apm );
    return 0;
}
#endif
