/*
 * GSysInfo Applet
 *  - A GNOME panel applet to display various system information.
 *  Copyright (C) 2000-2002 Jason D. Hildebrand
 *  - jason@alumni.uwaterloo.ca
 *  - http://www.opensky.ca/gsysinfo
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>
#include <dirent.h>
#include <string.h>
#include <time.h>
/* #include <config.h> */
#include <gnome.h>
#include <gdk/gdkx.h>

#include "gsysinfo.h"
#include "session.h"
#include "properties.h"
#include "sysinfo.h"
#include "gauge.h"


#if defined( DEBUG )
FILE * debugfile;
#endif


GSysInfoData * gd;


int
main (int argc, char ** argv)
{
  const gchar *goad_id;
  GtkWidget *applet;
  int            rc;

  /* Initialize i18n */
  bindtextdomain (PACKAGE, GNOMELOCALEDIR);
  textdomain (PACKAGE);

  #if defined( DEBUG )
  debugfile = fopen( DEBUG_FILENAME, "w" );
  if( debugfile == NULL ) {
    perror( DEBUG_FILENAME );
    return 1;
  }
  #endif

  applet_widget_init ("gsysinfo_applet", VERSION, argc, argv, NULL, 0, NULL);

  applet_factory_new ("gsysinfo_applet", NULL,
                 (AppletFactoryActivator) applet_start_new_applet);

  goad_id = goad_server_activation_id ();
  if (! goad_id) {
    return 1;
  }
  rc = sysinfo_init();
  if( rc != 0 ) {
    return 1;
  }

  /* Create the applet widget */
  applet = make_new_gsysinfo_applet (goad_id);
  if( applet == NULL ) {
      return 1;
  }

  /* Run... */

  applet_widget_gtk_main ();

  #if defined( DEBUG )
  fclose( debugfile );
  #endif

  return 0;
} /* main */


static void
set_tooltip_position( GSysInfoData * gd )
{
  // this code was extracted from gtktooltip.c in libgtk.
  
  GtkWidget *widget;
  GtkStyle *style;
  gint gap, x, y, w, h, scr_w, scr_h, baseline_skip;
  GList *el;

  widget = gd->applet;

  scr_w = gdk_screen_width ();
  scr_h = gdk_screen_height ();

  w = GTK_WIDGET( gd->tooltipwin )->allocation.width;
  h = GTK_WIDGET( gd->tooltipwin )->allocation.height;

  gdk_window_get_pointer (NULL, &x, NULL, NULL);
  gdk_window_get_origin (widget->window, NULL, &y);
  if (GTK_WIDGET_NO_WINDOW (widget))
    y += widget->allocation.y;

  x -= ((w >> 1) + 4);

  if ((x + w) > scr_w)
    x -= (x + w) - scr_w;
  else if (x < 0)
    x = 0;

  if ((y + h + widget->allocation.height + 4) > scr_h)
    y = y - h - 4;
  else
    y = y + widget->allocation.height + 4;

  gdk_window_move( GTK_WIDGET( gd->tooltipwin )->window, x, y);
}

char * network_units[] =
{
    "    %s: %.0f bytes/sec\n",
    "    %s: %.1f KB/sec\n",
    "    %s: %.1f MB/sec\n",
    "    %s: %.1f GB/sec\n",
    "    %s: %.1f TB/sec\n"
};

void calc_network_rate( struct netinfo * netinfo, double * rate, int * unit ) 
{
    double bps;

    bps = ( netinfo->sent_bytes - netinfo->old_sent_bytes + netinfo->received_bytes - netinfo->old_received_bytes ) / ( (double)gd->timeout_t / (double)1000 );

    *unit = 0;
    while( bps > 1024 ) {
        *unit += 1;
        bps /= 1024;
    }
    *rate = bps;
}

void update_tooltip( GSysInfoData *    gd )
{
    struct load *            curload = &(gd->curload);
    struct meminfo *  mem = &(gd->memstats[0]);

    int                user, system, nice, idle;
    int                buffmem, normalmem, freemem;
    int                normalperc, buffperc, freeperc;
    char               buffer[256];
    int                n;

    strcpy( gd->tooltip, "" );

    if( gd->need_load || gd->tooltipall ) {
        g_snprintf( buffer, 256, "Load: %.2f     \n", gd->loadavg );
        strcat( gd->tooltip, buffer );
    }

    if( gd->need_cpu || gd->tooltipall ) {
        user = curload->user * 100 / curload->total;
        system = curload->system * 100 / curload->total;
        nice = curload->nice * 100 / curload->total;
        idle = 100 - user - system - nice;

        g_snprintf( buffer, 256, 
                    "CPU:\n    User: %d%%\n    Kernel: %d%%\n    Nice: %d%%\n    Idle: %d%%\n",
                   user, system, nice, idle );
        strcat( gd->tooltip, buffer );
    }

    if( gd->need_meminfo ) {
        buffmem = mem->cache + mem->buffers;
        normalmem = mem->total - mem->free - buffmem;
        normalperc = normalmem * 100 / mem->total;
        buffperc = buffmem * 100 / mem->total;
        freeperc = mem->free * 100 / mem->total;
        normalmem /= 1024;
        buffmem /= 1024;
        freemem = mem->free / 1024;

        g_snprintf( buffer, 256, 
           "Memory:\n    Used: %d MB (%d%%)\n    Cached: %d MB (%d%%)\n    Free: %d MB (%d%%)\n"
           "Swap usage: %d%%\n",
               normalmem, normalperc,
               buffmem, buffperc,
               freemem, freeperc,
               gd->swap );
        strcat( gd->tooltip, buffer );
    }

    if( (gd->need_battery || gd->tooltipall ) && gd->battinfo.has_apm ) {
        g_snprintf( buffer, 256, 
           "Battery: %d%% (%s)\n",
           gd->battinfo.battery_perc, gd->battinfo.acstatus & AC_ONLINE ? "Online" : "Offline" );
        strcat( gd->tooltip, buffer );
    }

    if( gd->need_network || gd->tooltipall ) {
        int unit;
        double rate;
        strcat( gd->tooltip, "Network:\n" );
        for( int i = 0; i < gd->numnetdevs ; i++ ) {
            if( strcmp( gd->netinfo[i].iface, "lo" ) == 0 ) {
                continue;
            }
            calc_network_rate( &(gd->netinfo[i]), &rate, &unit );
            g_snprintf( buffer, 256, network_units[unit], gd->netinfo[i].iface, rate );
            strcat( gd->tooltip, buffer );
        }
    }
    // nix the last newline
    n = strlen( gd->tooltip );
    if( n > 0 ) {
        gd->tooltip[ n - 1 ] = '\0';
    }
    gtk_label_set_text( GTK_LABEL( gd->tooltiplabel ), gd->tooltip );
}


// get the current data (loadavg, load, mem, swap), and update each of the
// gauges
gint
gsysinfo_update (gpointer data)
{
    GSysInfoData *    gd = (GSysInfoData *)data;
    gboolean need_all = gd->tooltip_update && gd->tooltipall;

    // get the load and loag average, if we need this info
    if( gd->need_cpu || need_all ) {
        get_cpu ( &(gd->curload) );
    }
    if( gd->need_load || need_all ) {
        gd->loadavg = get_load();
    }

    // get memory information if necessary
    if( gd->need_meminfo || need_all ) {
       get_meminfo( &gd->numswapfiles, gd->memstats );
    }

    if( gd->need_battery || need_all ) {
        get_apm_status( &gd->battinfo );
    }

    if( gd->need_network || need_all ) {
        gd->numnetdevs = get_net_status( gd->netinfo );
    }

    // cause each widget to draw itself
    for( int i = 0; i < gd->num_gauges; i++ ) {
        gd->gauges[i]->Draw( gd, gd->bar_breadth, gd->bar_depth );
    }

    if( gd->tooltip_update ) {
        update_tooltip( gd );
    }
  
    /* Update the display. */
    // it's safe to use the zero'th area, since we'll guarantee
    // that there's always at least one gauge
    expose_handler (gd->gauges[0]->area, NULL, gd);
  
    return TRUE;
} /* gsysinfo_update */





/*
 * This function is called whenever a portion of the
 * applet window has been exposed and needs to be redrawn.  In this
 * function, we just blit the appropriate portion of the pixmap onto the window.
 *
 */
gint
expose_handler (GtkWidget * ignored, GdkEventExpose * expose,
                gpointer data)
{
    GSysInfoData * gd = (GSysInfoData *)data;

    // if we're not setup, don't try to draw anything
    if (!gd->setup) {
        return FALSE;
    }

    // display each of the gauges
    for( int i = 0; i < gd->num_gauges; i++ ) {
        gd->gauges[i]->Display( gd );
    }

    return FALSE; 
} /* expose_handler */


/* This handler gets called whenever the panel changes orientations.
   When the applet is set up, we get an initial call, too.  We don't
   actually do anything differently for vertical displays.  Not yet,
   anyways. */
gint
orient_handler (GtkWidget * w, PanelOrientType o, gpointer data)
{
  GSysInfoData * gd = (GSysInfoData *)data;
  gboolean vertical = (o == ORIENT_UP) || (o == ORIENT_DOWN);

  if (vertical != gd->vertical) {
    gd->vertical = vertical;
  }

  return FALSE;
} /* orient_handler */

static void tooltip_activate(GSysInfoData *gd) { 
    gint x,y;

    // remove the timeout
    gtk_timeout_remove (gd->tooltip_timeout );
    gd->tooltip_timeout = 0;

    // display the tooltip
    gd->tooltip_update = TRUE;
    gtk_widget_realize( gd->tooltipwin );
    set_tooltip_position( gd );
    gtk_widget_show( gd->tooltipwin );
}

gint
pointer_handler (GtkWidget * w, GdkEvent *event, gpointer data)
{
    GSysInfoData * gd = (GSysInfoData *)data;
    GdkEventType type;
  
    type = event->type;
    if( type == GDK_ENTER_NOTIFY ) {
        // pointer entered window
        // set timer.  When callback is called the tooltip will be activated
        // (unless the pointer leaves the window in the meantime and the
        // timeout is removed)
        if (gd->tooltip_timeout != 0 ) {
            // remove the old timeout if there was one
            gtk_timeout_remove (gd->tooltip_timeout );
            gd->tooltip_timeout = 0;
        }
        gd->tooltip_timeout = gtk_timeout_add (gd->tooltip_delay, (GtkFunction) tooltip_activate, gd);
    } else if( type == GDK_LEAVE_NOTIFY ) {
        if (gd->tooltip_timeout != 0 ) {
            // remove the old timeout if there was one
            gtk_timeout_remove (gd->tooltip_timeout );
            gd->tooltip_timeout = 0;
        }
        gd->tooltip_update = FALSE;
        gtk_widget_hide( gd->tooltipwin );
    }
    return FALSE;
} 

// the purpose of this function is to determine where 
// each of the indicators should be placed within the applet,
// according to the properties
void next_indicator_placement( GSysInfoData * gd, 
                       int * numbars, 
                       int * x, 
                       int * y )
{
    int bars_per_col = gd->num_gauges / gd->num_columns;
    if( ( gd->num_gauges % gd->num_columns ) > 0 ) {
        bars_per_col++;
    }

    if( *numbars == 0 ) {
        *x = 0;
        *y = 0;
        *numbars += 1;
    } else if( *numbars >= bars_per_col ) {
        *x += gd->bar_breadth + 4;
        *y = 0;
        *numbars = 1;
    } else {
        *y += gd->bar_depth + gd->gap_depth;
        *numbars += 1;
    }
}

// this function places each gauge within the applet,
// according to how many columns will be used
void configure_layout( GSysInfoData * gd ) {
    int curx, cury, pixmap_y;
    int numbars;
  
    curx = 0; 
    cury = 0; 
    numbars = 0;
    pixmap_y = 0;

    for( int i = 0; i < gd->num_gauges; i++ ) {
        next_indicator_placement( gd, &numbars, &curx, &cury );
        gtk_fixed_move( gd->fixed, gd->gauges[i]->frame, curx, cury );
        gtk_widget_show_all( gd->gauges[i]->frame );

        // tell the gauge which part of the pixmap to draw on
        gd->gauges[i]->SetPixmapPos( 0, pixmap_y );
        pixmap_y += gd->bar_depth;
    }
}


GtkWidget *
applet_start_new_applet (const gchar *goad_id, const char **params,
                 int nparams)
{
  return make_new_gsysinfo_applet (goad_id);
} /* applet_start_new_applet */

/* This is the function that actually creates the display widgets */
GtkWidget *
make_new_gsysinfo_applet (const gchar *goad_id) {
    gchar * param = "gsysinfo_applet";
    gd = g_new0 (GSysInfoData, 1);

    gd->applet = applet_widget_new (goad_id);
    gtk_widget_set_usize (gd->applet, 0, 0 ); 

    if (gd->applet == NULL) {
        return( NULL );
    }
    gd->setup = FALSE;

    /*
     * Load all the saved session parameters (or the defaults if none
     * exist).
     */
    // first load all of the default values
    gsysinfo_session_defaults (gd);

    // load values from config file if they exist
    if ( (APPLET_WIDGET (gd->applet)->privcfgpath) && *(APPLET_WIDGET (gd->applet)->privcfgpath)) {
      gsysinfo_session_load (APPLET_WIDGET (gd->applet)->privcfgpath, gd);
    }
    determine_needed_info( gd );

    // create container to hold the bars
    gd->fixed = (GtkFixed *) gtk_fixed_new();

    // insert the gauges into the container, and set the events
    for( int i = 0 ; i < gd->num_gauges; i++ ) {
        gd->gauges[i]->add_to_container( gd->fixed );
        gtk_signal_connect (GTK_OBJECT (gd->gauges[i]->area), "expose_event",
                   (GtkSignalFunc)expose_handler, gd);
        gtk_widget_set_events (gd->gauges[i]->area, GDK_EXPOSURE_MASK );
    }

  
    /* This will let us know when the panel changes orientation */
    gtk_signal_connect (GTK_OBJECT (gd->applet), "change_orient",
                    GTK_SIGNAL_FUNC (orient_handler),
                    gd);
  
    gd->tooltip_update = TRUE;
    gtk_signal_connect (GTK_OBJECT (gd->applet), "enter_notify_event",
                    GTK_SIGNAL_FUNC (pointer_handler),
                    gd);
  
    gtk_signal_connect (GTK_OBJECT (gd->applet), "leave_notify_event",
                    GTK_SIGNAL_FUNC (pointer_handler),
                    gd);
  
    applet_widget_add (APPLET_WIDGET (gd->applet), GTK_WIDGET( gd->fixed ) );
  
    gtk_signal_connect (GTK_OBJECT (gd->applet), "save_session",
                    GTK_SIGNAL_FUNC (gsysinfo_session_save),
                    gd);

    applet_widget_register_stock_callback (APPLET_WIDGET (gd->applet),
                           "about",
                           GNOME_STOCK_MENU_ABOUT,
                           _("About..."),
                           about_cb,
                           gd);
  
    applet_widget_register_stock_callback (APPLET_WIDGET (gd->applet),
                           "properties",
                           GNOME_STOCK_MENU_PROP,
                           ("Properties..."),
                           create_properties_window,
                           gd);
  
    gtk_widget_show_all (gd->applet);
  
    configure_layout( gd );
    gsysinfo_set_size (gd);
  
    gsysinfo_create_gc (gd);
  
    //gd->tooltips = gtk_tooltips_new();
    gd->tooltipwin = gtk_window_new( GTK_WINDOW_POPUP );
    gd->tooltiplabel = gtk_label_new( "" );
    gtk_label_set_justify( GTK_LABEL( gd->tooltiplabel ), GTK_JUSTIFY_LEFT );


    gtk_widget_show( gd->tooltiplabel );
    gtk_container_add( GTK_CONTAINER( gd->tooltipwin ), gd->tooltiplabel );
  
    /* Nothing is drawn until this is set. */
    gd->setup = TRUE;
  
    /* Will schedule an event to update the display*/
    set_update_event( gd );
  
    return gd->applet;
} /* make_new_gsysinfo_applet */

void set_update_event(GSysInfoData *gd) { 
    // remove the old timeout if there was one
    if (gd->timeout != 0 ) {
      gtk_timeout_remove (gd->timeout);
      gd->timeout = 0;
    }
    gd->timeout = gtk_timeout_add (gd->timeout_t, (GtkFunction) gsysinfo_update, gd);
}



void
destroy_about (GtkWidget *w, gpointer data)
{
  GSysInfoData *gd = (GSysInfoData *)data;
} /* destroy_about */

void
about_cb (AppletWidget *widget, gpointer data)
{
  GSysInfoData *gd = (GSysInfoData *)data;
  char *authors[2];
  
  authors[0] = "Jason Hildebrand <jason@alumni.uwaterloo.ca>";
  authors[1] = NULL;

  gd->about_box =
    gnome_about_new (_("GSysInfo Applet"), VERSION,
                 _("Copyright (C) 2000-2002 Jason D. Hildebrand"),
                 (const char **) authors,
             _("This applet displays the system load average, "
               "current CPU usage, memory usage, and swap file usage, battery status "
               "(for laptops) and network traffic in the same style as the program xsysinfo "
               " by Gabor Herr.\n"

               "Use the preferences to add/remove gauges from the display or modify the "
               "colours."

               " \nLOAD AVERAGE:\n"
               "The load average gauge is subdivided into n sections, where "
               "n is the next largest whole number greater than the load average.  "
               "For example, a load of 2.3 would be represented with three sections.  "
               "The first two are coloured in completely, and the third section is 3/10 "
               "filled in.\n"

               " \nCPU USAGE:\n"
               "The CPU usage gauge has up to three sections.  White (empty) represents "
               "idle time. Light blue represents user time, medium blue represents system "
               "time, and dark blue represents time used by 'nice' processes.\n"

               " \nMEMORY USAGE:\n"

               "The memory gauge has two sections: normal memory use (light green), and "
               "memory used for cache and buffers (dark green).\n"

               " \nSWAP USAGE:\n"

               "The swap space gauge shows how much of the swap file(s) are currently "
               "in use.\n"

               " \nBATTERY STATUS:\n"

               "The battery status gauge shows the life remaining in your battery, and "
               "shows (by the colour) whether your laptop is currently online (red) or "
               "offline (blue).\n"

               " \nNETWORK STATUS:\n"

               "The battery status gauge shows the traffic through one of your network "
               "interfaces (configurable in the preferences).  Green indicates outgoing "
               "data and red indicates incoming data.\n"

               "\n \nThis applet comes with ABSOLUTELY NO WARRANTY.\n"
               "This is free software, and you are welcome to redistribute it "
               "under certain conditions.  "
               "See the LICENSE file for details.\n" ),
                 NULL);

  gtk_signal_connect (GTK_OBJECT (gd->about_box), "destroy",
                  GTK_SIGNAL_FUNC (destroy_about), gd);

  gtk_widget_show (gd->about_box);
} /* about_cb */


void
gsysinfo_set_size (GSysInfoData * gd)
{
    int n;


    // set sizes of the drawing areas
    for( int i = 0; i < gd->num_gauges; i++ ) {
        gtk_widget_set_usize (gd->gauges[i]->area, 
                      gd->bar_breadth, 
                      gd->bar_depth ); 
    }

    //gd->depth = gd->num_gauges * gd->bar_depth 
              //+ ( gd->num_gauges - 1 ) * gd->gap_depth;

    /*
     * If pixmaps have already been allocated, then free them here
     * before creating new ones.  */
    if (gd->pixmap != NULL) {
        gdk_pixmap_unref (gd->pixmap);
    }

    // FIXME: what do the first and last parameters of this function do?
    // we shouldn't really use the zero'th area...
    gd->pixmap = gdk_pixmap_new( gd->gauges[0]->area->window, 
                         gd->bar_breadth, 
                         gd->bar_depth * gd->num_gauges,
                                 gtk_widget_get_visual (gd->gauges[0]->area)->depth);

    // clear drawing area
    gd->bg_color->use( gd->gc );
    gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, 0, 0, gd->bar_breadth, gd->bar_depth * gd->num_gauges);
  
} /* gsysinfo_set_size */

void
gsysinfo_create_gc (GSysInfoData * gd)
{
    // FIXME: can we really count on the zero'th area being available?
    gd->gc = gdk_gc_new ( gd->applet->window );

    //gdk_gc_copy (gd->gc, gd->gauges[0]->area->style->white_gc);

    gdk_gc_set_function (gd->gc, GDK_COPY);
} /* gsysinfo_create_gc */

void determine_needed_info( GSysInfoData * gd )
{
    gd->need_cpu = 0;
    gd->need_load = 0;
    gd->need_meminfo = 0;
    gd->need_battery = 0;
    gd->need_network = 0;

    for( int i = 0 ; i < gd->num_gauges; i++ ) {
        switch( gd->gauges[i]->getType() ) {
        case LOAD_GAUGE: 
            gd->need_load = 1;
            break;

        case CPU_GAUGE: 
            gd->need_cpu = 1;
            break;

        case MEM_GAUGE: 
        case SWAP_GAUGE: 
            gd->need_meminfo = 1;
            break;

        case BATTERY_GAUGE: 
            gd->need_battery = 1;
            break;

        case NETWORK_GAUGE:
            gd->need_network = 1;
            break;
        }
    }
}
