/*
 * GSysInfo Applet
 *  - A GNOME panel applet to display various system information.
 *  Copyright (C) 2000-2002 Jason D. Hildebrand
 *  - jason@alumni.uwaterloo.ca
 *  - http://www.opensky.ca/gsysinfo
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

extern "C" {
#include <applet-widget.h>
}

#include "gauge.h"

char * GaugeNames[][2] = {
    { "Load Average", "Load" },
    { "CPU Usage", "CPU" },
    { "Memory Usage", "Memory" },
    { "Swap Usage", "Swap" },
    { "Battery Status", "Battery" },
    { "Network Activity", "Network" }
};

Gauge *        new_gauge( GaugeType type )
{
    switch( type ) {
    case LOAD_GAUGE :
        return( (Gauge *) new LoadGauge );
        break;
    case CPU_GAUGE: 
        return( (Gauge *) new CPUGauge );
        break;
    case MEM_GAUGE: 
        return( (Gauge *) new MemGauge );
        break;
    case SWAP_GAUGE: 
        return( (Gauge *) new SwapGauge );
        break;
    case BATTERY_GAUGE: 
        return( (Gauge *) new BatteryGauge);
        break;
    case NETWORK_GAUGE:
        return( (Gauge *) new NetworkGauge );
        break;
    }
}

Gauge::Gauge() :
    container( 0 ),
    area( NULL ),
    frame( NULL )
{
    // create a frame
    frame = gtk_frame_new( NULL );
    gtk_frame_set_shadow_type( GTK_FRAME( frame ), GTK_SHADOW_IN );

    // create the drawing area
    area = gtk_drawing_area_new();

    // put the drawing area inside the frame
    gtk_container_add( GTK_CONTAINER( frame ), area );

    // init colors to NULL, so that we know if they've been allocated
    for( int i = 0; i < max_colors_per_gauge; i++ ) {
        colors[i] = NULL;
    }
}

Gauge * Gauge::clone()
{
    Gauge * newgauge = new_gauge( type );
    for( int i = 0; colors[i] != NULL && i < max_colors_per_gauge; i++ ) {
        if( newgauge->colors[i] != NULL ) {
            delete newgauge->colors[i];
        }
        newgauge->colors[i] = colors[i]->clone();
    }
    return newgauge;
}

Gauge::~Gauge()
{
    for( int i = 0; i < max_colors_per_gauge; i++ ) {
        if( colors[i] != NULL ) {
            delete colors[i];
        }
    }

    // if the frame was added to a container, remove it now
    if( container ) {
        gtk_container_remove( GTK_CONTAINER( container ), frame );
    }
   
    // remove area from the frame
    gtk_container_remove( GTK_CONTAINER( frame ), area ); 

    // delete the frame, area
    gtk_object_unref( GTK_OBJECT( frame ) );
    gtk_object_unref( GTK_OBJECT( area ) );


}

const char * Gauge::getName( int nametype )
{
    return( GaugeNames[ type ][nametype] );
}

void Gauge::Display( GSysInfoData * gd )
{
    gdk_draw_pixmap( area->window, 
                     area->style->fg_gc[GTK_WIDGET_STATE (area)],
                     gd->pixmap, 
                     pixmap_x, pixmap_y, 
                     0, 0, 
                     gd->bar_breadth, gd->bar_depth);
}

void Gauge::add_to_container( GtkFixed * cont )
{
    container = cont;
    gtk_fixed_put( container, frame, 0, 0 );
}

GtkWidget * Gauge::new_proppage( GnomePropertyBox * propwin )
{
    GtkWidget * vb;

    vb = gtk_vbox_new ( FALSE, 2 );
    for( int i = 0; i < max_colors_per_gauge; i++ ) {
        if( colors[i] != NULL ) {
            gtk_box_pack_start( GTK_BOX( vb ),colors[i]->new_props(propwin ), TRUE, FALSE, 0);
        }
    }
    gtk_widget_show( vb );
    return vb;
}

void Gauge::save_session( char * prefix )
{
    char prop[100];

    for( int i = 0; i < max_colors_per_gauge; i++ ) {
        if( colors[i] != NULL ) {
            sprintf( prop, "%scolor%d", prefix, i );
            colors[i]->save_session( prop );
        }
    }
    
}

void Gauge::load_session( char * prefix )
{
    char prop[100];

    for( int i = 0; i < max_colors_per_gauge; i++ ) {
        if( colors[i] != NULL ) {
            sprintf( prop, "%scolor%d", prefix, i );
            colors[i]->load_session( prop );
        }
    }
}

LoadGauge::LoadGauge() 
{
    type = LOAD_GAUGE;
    colors[0] = new Color( "Load Average 0", "loadavg0", "#FFA0A0" );
    colors[1] = new Color( "Load Average 1", "loadavg1", "#FF6A6A" );
    colors[2] = new Color( "Load Average 2", "loadavg2", "#FF4500" );
    colors[3] = new Color( "Load Average 3", "loadavg3", "#FF3838" );
    colors[4] = new Color( "Load Average 4", "loadavg4", "#FFB5C5" );
    colors[5] = new Color( "Load Average 5", "loadavg5", "#FF6EB4" );
    colors[6] = new Color( "Load Average 6", "loadavg6", "#FF1493" );
    colors[7] = new Color( "Load Average 7", "loadavg7", "#fcc1ff" );
    colors[8] = new Color( "Load Average 8", "loadavg8", "#caadff" );
    colors[9] = new Color( "Load Average 9", "loadavg9", "#c2e1ff" );
}


void LoadGauge::Draw( GSysInfoData * gd, 
                      int bar_breadth, 
                      int bar_depth ) {
    int segments;
    int segsize;
    int segstart;
    int segend;
    int n;

    segments = ((int)(gd->loadavg)) + 1;
    segsize = bar_breadth / segments;

    /* Clear the graph pixmap */
    gd->bg_color->use( gd->gc );
    gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, pixmap_x, pixmap_y, 
                        bar_breadth, bar_depth );

    segend = 0;
    for( n = 0; n < segments - 1; n++ ) {
        segstart = segend;
        segend = (( bar_breadth * 10 ) / segments ) * ( n + 1 ) / 10;
        segsize = segend - segstart;
        colors[ n % 10 ]->use( gd->gc );
        gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, 
                            pixmap_x + segstart, pixmap_y, segsize - 1, 
                            bar_depth );
    }
  
    segstart = segend;
    segend = bar_breadth - 1;
    segsize = (int)((gd->loadavg * (double)10 - ( segments - 1 ) * 10 ) * (segend - segstart + 1 ) / 10 + 1);
    if( segstart + segsize - 1 > segend ) {
        segsize = segend - segstart + 1;
    }
    colors[ (segments - 1) % 10 ]->use( gd->gc );
    gdk_draw_rectangle (gd->pixmap, 
                        gd->gc, 
                        TRUE, 
                        pixmap_x + segstart,
                        pixmap_y, 
                        segsize,
                        bar_depth );
}


CPUGauge::CPUGauge()
{
    type = CPU_GAUGE ;
    colors[0] = new Color( "User CPU", "usercpu", "#00EEEE" );
    colors[1] = new Color( "System CPU", "systemcpu", "#5CACEE" );
    colors[2] = new Color( "Nice CPU", "nicecpu", "#0000FF" );
}


void CPUGauge::Draw( GSysInfoData * gd, 
                     int bar_breadth, 
                     int bar_depth ) 
{

    int part1, part2, part3;
    struct load * curload = &(gd->curload);

    /* Calculate the bounds */
    part1 = ( curload->user * bar_breadth + 1 ) / curload->total;
    part2 = ( curload->system * bar_breadth + 1 ) / curload->total;
    part3 = ( curload->nice * bar_breadth + 1 ) / curload->total;
  
    /* Clear the graph pixmap */
    gd->bg_color->use( gd->gc );
    gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, pixmap_x, pixmap_y, bar_breadth, bar_depth );

    /* Draw the load bar graph */
    colors[ LOADBAR ]->use( gd->gc );
    gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, pixmap_x, pixmap_y, part1, bar_depth );
  
    colors[ LOADBAR1 ]->use( gd->gc );
    gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, pixmap_x + part1, pixmap_y, part2, bar_depth );
  
    colors[ LOADBAR2 ]->use( gd->gc );
    gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, pixmap_x + part1 + part2, pixmap_y, part3, bar_depth );
  }
  
MemGauge::MemGauge() 
{
    type = MEM_GAUGE ;
    colors[0] = new Color ( "Normal Mem", "normalmem", "#00EE00" );
    colors[1] = new Color ( "Buffers/Cached Mem", "buffermem", "#008B00" );
}

void MemGauge::Draw( GSysInfoData * gd, 
                    int bar_breadth, 
                    int bar_depth ) 
{
    int normalmem, buffmem;
    struct meminfo * mem;
    int segstart;
    int        segend;
    int segsize;

    mem = &(gd->memstats[0]);
    buffmem = mem->cache + mem->buffers;
    normalmem = mem->total - mem->free - buffmem;

    // Clear the graph pixmap 
    gd->bg_color->use( gd->gc );
    gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, pixmap_x, pixmap_y, bar_breadth, bar_depth );

    // Draw the "normal used memory" bar
    segend = normalmem * bar_breadth / mem->total;
    segsize = segend - 1;
    colors[ MEMBAR0 ]->use( gd->gc );
    gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, pixmap_x, pixmap_y, segsize, bar_depth );

    // Draw the buffered + cached memory bar
    segstart = segend;
    segend = ( buffmem + normalmem ) * bar_breadth / mem->total;
    segsize = segend - segstart + 1;
    colors[ MEMBAR1 ]->use( gd->gc );
    gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, pixmap_x + segstart, pixmap_y, segsize, bar_depth );
}

SwapGauge::SwapGauge()
{
    type = SWAP_GAUGE;
    colors[0] = new Color( "Swap Used", "swap", "#FF6EB4" );
}

void SwapGauge::Draw( GSysInfoData * gd, int bar_breadth, int bar_depth ) {
    unsigned long totalspace = 0;
    unsigned long totalfree = 0;
    unsigned long totalused;
    int n;
    int segsize;
    #if defined( DEBUG )
    char buff[ 255 ];
    #endif
    
    for( n = 0; n < gd->numswapfiles ; n++ ) {
        totalspace += gd->memstats[ n + 1 ].total;
        totalfree += gd->memstats[ n + 1 ].free;
    }
    totalused = totalspace - totalfree;

    // Clear the graph pixmap 
    gd->bg_color->use( gd->gc );
    gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, pixmap_x, pixmap_y, bar_breadth, bar_depth );

    if( totalspace > 0 ) {
        segsize = totalused * bar_breadth / totalspace + 1;
        gd->swap = totalused * 100 / totalspace;
    } else {
        segsize = 0;
        gd->swap = 0;
    }
    colors[ SWAPBAR ]->use( gd->gc );
    gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, pixmap_x, pixmap_y, segsize, bar_depth );

    return;
}


BatteryGauge::BatteryGauge()
{
    type = BATTERY_GAUGE;
    colors[0] = new Color( "Online", "online", "#FF0000" );
    colors[1] = new Color( "Offline", "offline", "#0000FF" );
}

void BatteryGauge::Draw( GSysInfoData * gd, int bar_breadth, int bar_depth ) {
    int segsize;
    
    // Clear the graph pixmap 
    gd->bg_color->use( gd->gc );
    gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, pixmap_x, pixmap_y, bar_breadth, bar_depth );

    segsize = gd->battinfo.battery_perc * bar_breadth / 100 + 1;
    if( gd->battinfo.acstatus & AC_ONLINE ) {
        colors[ ONLINE ]->use( gd->gc );
    } else {
        colors[ OFFLINE ]->use( gd->gc );
    }
    gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, pixmap_x, pixmap_y, segsize, bar_depth );

    return;
}



NetworkGauge::NetworkGauge()
{
    type = NETWORK_GAUGE;
    index = -1;
    bandwidth = 16;
    ifaces = NULL;
    offline = FALSE;
    prop_win = NULL;
    
    // todo: choose a good default interface (look at existing interfaces) and
    // existing gauges
    strcpy( interface, "lo" );
    colors[0] = new Color( "Send", "send", "#00FF00" );
    colors[1] = new Color( "Receive", "receive", "#FF0000" );
    colors[2] = new Color( "Offline", "offline", "#3d79b5" );
}

NetworkGauge::~NetworkGauge()
{
    char * iface;
    while( ifaces != NULL ) {
        iface = (char *)g_list_nth_data( ifaces, 0 );
        free( iface );
        ifaces = g_list_remove( ifaces, iface );
    }
}

void NetworkGauge::Draw( GSysInfoData * gd, int bar_breadth, int bar_depth ) {
    struct netinfo * net = gd->netinfo;
    int part1;
    int        part2;
    int sent;
    int received;

    if( index == -1 || strcmp( net[index].iface, interface ) != 0 ) {
        FindInterface( gd );
        if( index == -1 ) {
            if( !offline ) {
                colors[ OFFLINE ]->use( gd->gc );
                gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, pixmap_x, pixmap_y, bar_breadth, bar_depth );
                offline = TRUE;
            }
            return;
        }
    }
    sent = net[index].sent_bytes - prev_sent;
    received = net[index].received_bytes - prev_received;
    prev_sent = net[index].sent_bytes;
    prev_received = net[index].received_bytes;

    // We don't know the "maximum bandwith" of the interface, so what we
    // display on the gauge is just "relative" traffic.  We keep track of the 
    // "Maximum Experienced Bandwidth", and calculate things based on that.
    // The -1 is so that the "maximum experienced bandwidth" will gradually sink with time,
    // so that bandwidth spikes don't blow everything out of proportion (and
    // make everything else too tiny)
    bandwidth = MAX( sent + received + 1, bandwidth - (bandwidth * 1 / 100) );
    if( bandwidth <= 0 ) {
        bandwidth = 1;
    }

    /* Calculate the bounds */
    part1 = ( sent * bar_breadth + 1 ) / bandwidth;
    part2 = ( received * bar_breadth + 1 ) / bandwidth;
  
    // if there was _any_ traffic, make sure it shows at least a little bit
    if( sent > 0 ) {
        part1 = MAX( part1, 1 );
    }
    if( received > 0 ) {
        part2 = MAX( part2, 1 );
    }

    /* Clear the graph pixmap */
    gd->bg_color->use( gd->gc );
    gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, pixmap_x, pixmap_y, bar_breadth, bar_depth );

    /* Draw the load bar graph */
    colors[ SEND ]->use( gd->gc );
    gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, pixmap_x, pixmap_y, part1, bar_depth );
  
    colors[ RECEIVE ]->use( gd->gc );
    gdk_draw_rectangle (gd->pixmap, gd->gc, TRUE, pixmap_x + part1, pixmap_y, part2, bar_depth );
}

void NetworkGauge::FindInterface( GSysInfoData * gd )
{
    struct netinfo * net = gd->netinfo;

    index = -1;
    for( int i = 0; i < gd->numnetdevs; i++ ) {
        if( strcmp( net[i].iface, interface ) == 0 ) {
            index = i;
            prev_sent = net[i].sent_bytes;
            prev_received = net[i].received_bytes;
            offline = FALSE;
            return;
        }
    }
}

void NetworkGauge::set_iface( char * newiface )
{
    strcpy( interface, newiface );
    index = -1;        // this causes the index to be redetermined
    if( prop_win != NULL ) {
        gnome_property_box_changed( prop_win );
    }
}

void iface_changed_cb( GtkWidget *widget, gpointer data )
{
    NetworkGauge * gauge = (NetworkGauge *)data;
    gchar * newtext = gtk_editable_get_chars( GTK_EDITABLE( widget ), 0 , -1 );
    gauge->set_iface( newtext );
    g_free( newtext );
}

GtkWidget * NetworkGauge::new_proppage( GnomePropertyBox * propwin )
{
    GtkWidget * vb;
    GtkWidget * colors;
    GtkWidget * editable;
    GtkWidget * hb;
    struct netinfo netinfo[MAX_NET_DEVS];
    int                numdevs;
    char *        iface;

    // store away the property box ptr, so we can set the prop box as 
    // modified later
    prop_win = propwin;

    // empty the (previous) list and free all strings
    while( ifaces != NULL ) {
        iface = (char *)g_list_nth_data( ifaces, 0 );
        free( iface );
        ifaces = g_list_remove( ifaces, iface );
    }

    vb = gtk_vbox_new ( FALSE, 10 );
    numdevs = get_net_status( netinfo );
    for( int i = 0; i < numdevs; i++ ) {
        ifaces = g_list_append( ifaces, strdup( netinfo[i].iface ) );
    }
    hb = gtk_hbox_new( FALSE, 10 );
    combo = gtk_combo_new();
    gtk_combo_set_popdown_strings( GTK_COMBO( combo ), ifaces );
                                    
    gtk_box_pack_start( GTK_BOX( hb ), gtk_label_new( "Network interface" ), TRUE, FALSE, 0 );
    gtk_box_pack_start( GTK_BOX( hb ), combo, TRUE, FALSE, 0 );
    gtk_box_pack_start( GTK_BOX( vb ), hb, TRUE, FALSE, 0 );

    // initialize the text to the current interface
    gint position = 0; 
    editable = GTK_COMBO( combo )->entry;
    gtk_editable_delete_text( GTK_EDITABLE( editable ), 0 , -1 );
    gtk_editable_insert_text( GTK_EDITABLE( editable ), 
                              interface,
                              strlen( interface ), 
                              &position );

    // call callback function whenever text changes
    gtk_signal_connect (GTK_OBJECT ( editable ), "changed",
                        GTK_SIGNAL_FUNC (iface_changed_cb), this );
    
    colors = Gauge::new_proppage( propwin );
    gtk_box_pack_start_defaults( GTK_BOX( vb ), colors );
    gtk_widget_show_all( vb );
    return vb;
}

Gauge * NetworkGauge::clone()
{
    NetworkGauge * gauge = (NetworkGauge *)Gauge::clone();
    strcpy( gauge->interface, interface );
    gauge->index = index;
    return( (Gauge *)gauge );
}


void NetworkGauge::save_session( char * prefix )
{
    char prop[100];

    sprintf( prop, "%sinterface", prefix );
    gnome_config_set_string( prop, interface );
    Gauge::save_session( prefix );
    
}

void NetworkGauge::load_session( char * prefix )
{
    char prop[100];
    char * iface = NULL;

    sprintf( prop, "%sinterface=%s", prefix,interface );
    iface = gnome_config_get_string_with_default( prop, NULL );
    if( iface != NULL ) {
        strcpy( interface, iface );
        free( iface );
        index = -1;        // this causes the index to be recalculated
    }
    Gauge::load_session( prefix );
}

const char * NetworkGauge::getName( int nametype )
{
    strcpy( gaugename, GaugeNames[ type ][ nametype ] );
    strcat( gaugename, ": " );
    strcat( gaugename, interface );
    return( gaugename );
}
