
/*
 * sysinfo.c		- get kernel info
 *
 */

/* 
 * Written by Gabor Herr <herr@iti.informatik.th-darmstadt.de>.
 *
 * Copyright (c) 1992, 1993 by Gabor Herr, all rights reserved.
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that may name is not used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission. I make no representations
 * about the suitability of this software for any purpose. It is
 * provided "as is" without express or implied warranty.
 */

//
// modified in 1999, 2000 by Jason Hildebrand for use with gsysinfo 
//

/* $Id: sysinfo.c,v 1.1 2000/06/23 05:20:05 jdhildeb Exp $ */
 
#include "sysinfo.h"
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>

static int meminfo_fd;
static int load_fd;
static int loadavg_fd;
static int swaps_fd;
#if defined( DEBUG )
extern FILE * debugfile;
#endif

static char buffer[1024];

static void reread( int fd, char *message )
{
	ssize_t len;
	int	rc;

	rc = lseek( fd, 0L, 0 );
	if( rc != 0 ) {
	    perror( message );
	    exit( 1 );
	}
	len = read( fd, buffer, sizeof(buffer) - 1 );
	if( len <= 0 ) {
	    perror( message );
	    exit( 1 );
	};
	buffer[ len ] = '\0';	// stick a null char at the end of the file, so that we can
				// test for it later
	return;
}

static int getvalues( struct meminfo * result ) {
    char * bufptr = strchr( buffer, '\n' );
    char * endptr;
    int numvalues = 0;
    unsigned long * target;
    unsigned long value;

    bufptr++;  // advance past the newline

    // now try to recognize one of the tags: MemTotal, MemFree, MemShared,
    // Buffers, Cached, SwapTotal, SwapFree very quickly
    while( *bufptr != '\0' && numvalues < 7 ) {
	target = NULL;
	switch( *bufptr ) {
	case 'M':
	    switch( bufptr[ 3 ] ) {
	    case 'T':
		if( strncmp( bufptr, "MemTotal:", 9 ) == 0 ) {
		    bufptr += 9;
		    target = &result->total;
		}
		break;
	    case 'F':
		if( strncmp( bufptr, "MemFree:", 8 ) == 0 ) {
		    bufptr += 8;
		    target = &result->free;
		}
		break;
	    case 'S':
		if( strncmp( bufptr, "MemShared:", 10 ) == 0 ) {
		    bufptr += 10;
		    target = &result->shared;
		}
		break;
	    default:
		break;
	    }
	    break;
	case 'B':
	    if( strncmp( bufptr, "Buffers:", 8 ) == 0 ) {
		bufptr += 8;
		target = &result->buffers;
	    }
	    break;
	case 'C':
	    if( strncmp( bufptr, "Cached:", 7 ) == 0 ) {
		bufptr += 7;
		target = &result->cache;
	    }
	    break;
	case 'S':
	    if( bufptr[ 4 ] == 'T' ) {
		if( strncmp( bufptr, "SwapTotal:", 10 ) == 0 ) {
		    bufptr += 10;
		    target = &result[1].total;
		}
	    } else if( bufptr[ 4 ] == 'F' ) {
		if( strncmp( bufptr, "SwapFree:", 9 ) == 0 ) {
		    bufptr += 9;
		    target = &result[1].free;
		}
	    }
	    break;
	default:
	    break;
	}

	if( target != NULL ) {
	    // if we found a value that we want, then read it and assign it
	    // to the target struct member
	    value = strtoul( bufptr, &endptr, 10 );
	    if( bufptr != endptr ) {
		*target = value;
		numvalues += 1;
	    }
	}
	bufptr = strchr( bufptr, '\n' );
	bufptr++;
    }
    return( numvalues - 7 );
}

void get_meminfo( int *swapdevs, struct meminfo *result )
{
	int i;
	char *bufptr;
	int res;
	int sw_size, sw_used;
	int rc;

	reread( meminfo_fd, "get_meminfo");

	/* Try new /proc/meminfo format first */
	*swapdevs = 1;
    
	rc = getvalues( result );
	if( rc == 0 ) {
	    if (swaps_fd > 0) {

		reread( swaps_fd, "get_meminfo");
		bufptr = buffer;

		for ( i = 1; i < MAX_SWAPFILES; i++ ) {

			bufptr = strchr( bufptr, '\n' );
			bufptr++;

			if ( *bufptr == '\0' ) {
			    break;
			}
			res = sscanf( bufptr, "%*s %*s %d %d", &sw_size, &sw_used);
			if( res != 2 ) {
			    break;
			}

			result[i].total = sw_size;
			result[i].free = sw_size - sw_used;
		}

		*swapdevs = i - 1;  
	    }
	} else {
	    /* Fall back to old format */
	    bufptr = strchr( buffer, '\n' ) + 1;
	    sscanf( bufptr, "Mem: %u %*u %u %u %u",
		    &result[0].total,
		    &result[0].free,
		    &result[0].shared,
		    &result[0].cache );
	    result[0].buffers = 0;

	    for ( i = 1; i < MAX_SWAPFILES; i++ ) {
		    bufptr = strchr( bufptr, '\n' ) + 1;
		    if ( *bufptr == '\0' ||
			 (res = sscanf( bufptr, "Swap: %u %*u %u",
					&result[i].total,
					&result[i].free )) != 2 )
			    break; 
	    }
	    *swapdevs = i - 1;  
	}
}

double get_loadavg( void )
{
	double load;

	reread( loadavg_fd, "get_loadavg");

	sscanf( buffer, "%lf", &load );

	return load;
}

void get_load(struct load * result)
{
	static struct load last_load = { 0, 0, 0, 0, 0 };
	struct load curr_load;

	reread( load_fd, "get_load" );
	sscanf( buffer, "%*s %lu %lu %lu %lu\n", &curr_load.user,
		&curr_load.nice, &curr_load.system, &curr_load.idle );
	curr_load.total = curr_load.user + curr_load.nice + curr_load.system +
		curr_load.idle;

	result->total  = curr_load.total  - last_load.total;
	result->user   = curr_load.user   - last_load.user;
	result->nice   = curr_load.nice   - last_load.nice;
	result->system = curr_load.system - last_load.system;
	result->idle   = curr_load.idle   - last_load.idle;
	last_load.total  = curr_load.total;
	last_load.user   = curr_load.user;
	last_load.nice   = curr_load.nice;
	last_load.system = curr_load.system;
	last_load.idle   = curr_load.idle;
}

int sysinfo_init( void )
{
	if ((meminfo_fd = open("/proc/meminfo",O_RDONLY)) < 0) {
		perror("/proc/meminfo");
		return 1;
	}
	if ((loadavg_fd = open("/proc/loadavg",O_RDONLY)) < 0) {
		perror("/proc/loadavg");
		return 1;
	}
	if ((load_fd = open("/proc/stat",O_RDONLY)) < 0) {
		perror("/proc/stat");
		return 1;
	}
	/* Failing here is not critical. We can get the info from meminfo. */
	swaps_fd = open("/proc/swaps",O_RDONLY);

	return 0;
}

